package brachy.modularui.drawable;

import brachy.modularui.GTRenderTypes;
import brachy.modularui.ModularUI;
import brachy.modularui.client.GuiSpriteManager;
import brachy.modularui.drawable.text.TextRenderer;
import brachy.modularui.screen.RichTooltip;
import brachy.modularui.screen.event.RichTooltipEvent;
import brachy.modularui.screen.viewport.GuiContext;
import brachy.modularui.screen.viewport.ModularGuiContext;
import brachy.modularui.utils.Alignment;
import brachy.modularui.utils.Color;
import brachy.modularui.utils.FormattingUtil;
import brachy.modularui.utils.RectangleF;
import brachy.modularui.widget.sizer.Area;

import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.GuiGraphics;
import net.minecraft.client.gui.screens.inventory.tooltip.ClientTooltipComponent;
import net.minecraft.client.renderer.GameRenderer;
import net.minecraft.client.renderer.LightTexture;
import net.minecraft.client.renderer.RenderType;
import net.minecraft.client.renderer.entity.EntityRenderDispatcher;
import net.minecraft.client.renderer.texture.MissingTextureAtlasSprite;
import net.minecraft.client.renderer.texture.TextureAtlasSprite;
import net.minecraft.network.chat.Component;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.util.FormattedCharSequence;
import net.minecraft.util.Mth;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.LivingEntity;
import net.minecraft.world.inventory.InventoryMenu;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.level.material.Fluid;
import com.mojang.blaze3d.platform.GlStateManager;
import com.mojang.blaze3d.platform.Lighting;
import com.mojang.blaze3d.systems.RenderSystem;
import com.mojang.blaze3d.vertex.BufferBuilder;
import com.mojang.blaze3d.vertex.DefaultVertexFormat;
import com.mojang.blaze3d.vertex.Tesselator;
import com.mojang.blaze3d.vertex.VertexConsumer;
import com.mojang.blaze3d.vertex.VertexFormat;
import net.minecraftforge.api.distmarker.Dist;
import net.minecraftforge.api.distmarker.OnlyIn;
import net.minecraftforge.client.event.RenderTooltipEvent;
import net.minecraftforge.client.extensions.common.IClientFluidTypeExtensions;
import net.minecraftforge.common.MinecraftForge;
import net.minecraftforge.fluids.FluidStack;

import org.jetbrains.annotations.Nullable;
import org.joml.Matrix4d;
import org.joml.Matrix4f;
import org.joml.Quaternionf;
import org.joml.Vector3d;

import java.util.List;
import java.util.function.BiConsumer;

import static brachy.modularui.drawable.UITexture.GUI_TEXTURE_ID_CONVERTER;
import static net.minecraft.util.Mth.HALF_PI;
import static net.minecraft.util.Mth.TWO_PI;

public class GuiDraw {

    private static final TextRenderer textRenderer = new TextRenderer();

    public static void drawRect(GuiGraphics graphics, float x0, float y0, float w, float h, int color) {
        Matrix4f pose = graphics.pose().last().pose();
        VertexConsumer builder = graphics.bufferSource().getBuffer(RenderType.guiOverlay());
        drawRectRaw(builder, pose, x0, y0, x0 + w, y0 + h, color);
    }

    public static void drawHorizontalGradientRect(GuiGraphics graphics, float x0, float y0, float w, float h,
                                                  int colorLeft, int colorRight) {
        drawRect(graphics, x0, y0, w, h, colorLeft, colorRight, colorLeft, colorRight);
    }

    public static void drawVerticalGradientRect(GuiGraphics graphics, float x0, float y0, float w, float h,
                                                int colorTop, int colorBottom) {
        drawRect(graphics, x0, y0, w, h, colorTop, colorTop, colorBottom, colorBottom);
    }

    public static void drawRect(GuiGraphics graphics, float x0, float y0, float w, float h,
                                int colorTL, int colorTR, int colorBL, int colorBR) {
        Matrix4f pose = graphics.pose().last().pose();
        VertexConsumer bufferbuilder = graphics.bufferSource().getBuffer(RenderType.guiOverlay());

        float x1 = x0 + w, y1 = y0 + h;
        bufferbuilder.vertex(pose, x0, y0, 0.0f)
                .color(Color.getRed(colorTL), Color.getGreen(colorTL), Color.getBlue(colorTL), Color.getAlpha(colorTL))
                .endVertex();
        bufferbuilder.vertex(pose, x0, y1, 0.0f)
                .color(Color.getRed(colorBL), Color.getGreen(colorBL), Color.getBlue(colorBL), Color.getAlpha(colorBL))
                .endVertex();
        bufferbuilder.vertex(pose, x1, y1, 0.0f)
                .color(Color.getRed(colorBR), Color.getGreen(colorBR), Color.getBlue(colorBR), Color.getAlpha(colorBR))
                .endVertex();
        bufferbuilder.vertex(pose, x1, y0, 0.0f)
                .color(Color.getRed(colorTR), Color.getGreen(colorTR), Color.getBlue(colorTR), Color.getAlpha(colorTR))
                .endVertex();
    }

    public static void drawRectRaw(VertexConsumer buffer, Matrix4f pose, float x0, float y0, float x1, float y1,
                                   int color) {
        int r = Color.getRed(color);
        int g = Color.getGreen(color);
        int b = Color.getBlue(color);
        int a = Color.getAlpha(color);
        drawRectRaw(buffer, pose, x0, y0, x1, y1, r, g, b, a);
    }

    public static void drawRectRaw(VertexConsumer buffer, Matrix4f pose, float x0, float y0, float x1, float y1, int r,
                                   int g, int b, int a) {
        buffer.vertex(pose, x0, y0, 0.0f).color(r, g, b, a).endVertex();
        buffer.vertex(pose, x0, y1, 0.0f).color(r, g, b, a).endVertex();
        buffer.vertex(pose, x1, y1, 0.0f).color(r, g, b, a).endVertex();
        buffer.vertex(pose, x1, y0, 0.0f).color(r, g, b, a).endVertex();
    }

    public static void drawCircle(GuiGraphics graphics, float x0, float y0, float diameter, int color, int segments) {
        drawEllipse(graphics, x0, y0, diameter, diameter, color, color, segments);
    }

    public static void drawCircle(GuiGraphics graphics, float x0, float y0, float diameter,
                                  int centerColor, int outerColor, int segments) {
        drawEllipse(graphics, x0, y0, diameter, diameter, centerColor, outerColor, segments);
    }

    public static void drawEllipse(GuiGraphics graphics, float x0, float y0, float w, float h,
                                   int color, int segments) {
        drawEllipse(graphics, x0, y0, w, h, color, color, segments);
    }

    public static void drawEllipse(GuiGraphics graphics, float x0, float y0, float w, float h,
                                   int centerColor, int outerColor, int segments) {
        Matrix4f pose = graphics.pose().last().pose();
        VertexConsumer bufferbuilder = graphics.bufferSource().getBuffer(GTRenderTypes.guiOverlayTriangleFan());

        float x_2 = x0 + w / 2f, y_2 = y0 + h / 2f;
        // start at center
        bufferbuilder.vertex(pose, x_2, y_2, 0.0f)
                .color(Color.getRed(centerColor), Color.getGreen(centerColor), Color.getBlue(centerColor),
                        Color.getAlpha(centerColor))
                .endVertex();
        int a = Color.getAlpha(outerColor), r = Color.getRed(outerColor), g = Color.getGreen(outerColor),
                b = Color.getBlue(outerColor);
        float incr = TWO_PI / segments;
        for (int i = 0; i <= segments; i++) {
            float angle = incr * i;
            float x = Mth.sin(angle) * (w / 2) + x_2;
            float y = Mth.cos(angle) * (h / 2) + y_2;
            bufferbuilder.vertex(x, y, 0.0f).color(r, g, b, a).endVertex();
        }
        RenderSystem.disableBlend();
    }

    public static void drawRoundedRect(GuiGraphics graphics, float x0, float y0, float w, float h,
                                       int color, int cornerRadius, int segments) {
        drawRoundedRect(graphics, x0, y0, w, h, color, color, color, color, cornerRadius, segments);
    }

    public static void drawVerticalGradientRoundedRect(GuiGraphics graphics, float x0, float y0, float w, float h,
                                                       int colorTop, int colorBottom, int cornerRadius, int segments) {
        drawRoundedRect(graphics, x0, y0, w, h, colorTop, colorTop, colorBottom, colorBottom, cornerRadius, segments);
    }

    public static void drawHorizontalGradientRoundedRect(GuiGraphics graphics, float x0, float y0, float w, float h,
                                                         int colorLeft, int colorRight,
                                                         int cornerRadius, int segments) {
        drawRoundedRect(graphics, x0, y0, w, h, colorLeft, colorRight, colorLeft, colorRight, cornerRadius, segments);
    }

    public static void drawRoundedRect(GuiGraphics graphics, float x0, float y0, float w, float h,
                                       int colorTL, int colorTR, int colorBL, int colorBR,
                                       int cornerRadius, int segments) {
        Matrix4f pose = graphics.pose().last().pose();
        VertexConsumer bufferbuilder = graphics.bufferSource().getBuffer(GTRenderTypes.guiOverlayTriangleFan());

        float x1 = x0 + w, y1 = y0 + h;
        int color = Color.average(colorBL, colorBR, colorTR, colorTL);
        // start at center
        bufferbuilder.vertex(pose, x0 + w / 2f, y0 + h / 2f, 0.0f)
                .color(Color.getRed(color), Color.getGreen(color), Color.getBlue(color), Color.getAlpha(color))
                .endVertex();
        // left side
        bufferbuilder.vertex(pose, x0, y0 + cornerRadius, 0.0f)
                .color(Color.getRed(colorTL), Color.getGreen(colorTL), Color.getBlue(colorTL), Color.getAlpha(colorTL))
                .endVertex();
        bufferbuilder.vertex(pose, x0, y1 - cornerRadius, 0.0f)
                .color(Color.getRed(colorBL), Color.getGreen(colorBL), Color.getBlue(colorBL), Color.getAlpha(colorBL))
                .endVertex();
        // bottom left corner
        for (int i = 1; i <= segments; i++) {
            float x = x0 + cornerRadius - Mth.cos(HALF_PI / segments * i) * cornerRadius;
            float y = y1 - cornerRadius + Mth.sin(HALF_PI / segments * i) * cornerRadius;
            bufferbuilder.vertex(pose, x, y, 0.0f)
                    .color(Color.getRed(colorBL), Color.getGreen(colorBL), Color.getBlue(colorBL),
                            Color.getAlpha(colorBL))
                    .endVertex();
        }
        // bottom side
        bufferbuilder.vertex(pose, x1 - cornerRadius, y1, 0.0f)
                .color(Color.getRed(colorBR), Color.getGreen(colorBR), Color.getBlue(colorBR), Color.getAlpha(colorBR))
                .endVertex();
        // bottom right corner
        for (int i = 1; i <= segments; i++) {
            float x = x1 - cornerRadius + Mth.sin(HALF_PI / segments * i) * cornerRadius;
            float y = y1 - cornerRadius + Mth.cos(HALF_PI / segments * i) * cornerRadius;
            bufferbuilder.vertex(pose, x, y, 0.0f)
                    .color(Color.getRed(colorBR), Color.getGreen(colorBR), Color.getBlue(colorBR),
                            Color.getAlpha(colorBR))
                    .endVertex();
        }
        // right side
        bufferbuilder.vertex(pose, x1, y0 + cornerRadius, 0.0f)
                .color(Color.getRed(colorTR), Color.getGreen(colorTR), Color.getBlue(colorTR), Color.getAlpha(colorTR))
                .endVertex();
        // top right corner
        for (int i = 1; i <= segments; i++) {
            float x = x1 - cornerRadius + Mth.cos(HALF_PI / segments * i) * cornerRadius;
            float y = y0 + cornerRadius - Mth.sin(HALF_PI / segments * i) * cornerRadius;
            bufferbuilder.vertex(pose, x, y, 0.0f)
                    .color(Color.getRed(colorTR), Color.getGreen(colorTR), Color.getBlue(colorTR),
                            Color.getAlpha(colorTR))
                    .endVertex();
        }
        // top side
        bufferbuilder.vertex(pose, x0 + cornerRadius, y0, 0.0f)
                .color(Color.getRed(colorTL), Color.getGreen(colorTL), Color.getBlue(colorTL), Color.getAlpha(colorTL))
                .endVertex();
        // top left corner
        for (int i = 1; i <= segments; i++) {
            float x = x0 + cornerRadius - Mth.sin(HALF_PI / segments * i) * cornerRadius;
            float y = y0 + cornerRadius - Mth.cos(HALF_PI / segments * i) * cornerRadius;
            bufferbuilder.vertex(pose, x, y, 0.0f)
                    .color(Color.getRed(colorTL), Color.getGreen(colorTL), Color.getBlue(colorTL),
                            Color.getAlpha(colorTL))
                    .endVertex();
        }
        bufferbuilder.vertex(pose, x0, y0 + cornerRadius, 0.0f)
                .color(Color.getRed(colorTL), Color.getGreen(colorTL), Color.getBlue(colorTL), Color.getAlpha(colorTL))
                .endVertex();
    }

    /**
     * Set up the texture as a sampler differently depending on if it's part of the GUI atlas or not.
     * <p>
     * If the texture is part of the GUI atlas, the atlas will be used for drawing and the UV coordinates will be scaled to the atlas texture.<br>
     * If it's not part of the GUI atlas, nothing special is done.
     *
     * @param location the texture <i>file's</i> location, e.g. {@code "modularui:textures/gui/slot/item.png"}
     * @param u0       u0 UV coordinate
     * @param v0       v0 UV coordinate
     * @param u1       u1 UV coordinate
     * @param v1       v1 UV coordinate
     * @return If texture is in the GUI atlas, rescaled UV coordinates {@code u0, v0, u1, v1}.<br>
     * If not, the same coordinates that were passed in.
     */
    public static RectangleF setupTexture(ResourceLocation location, float u0, float v0, float u1, float v1) {
        TextureAtlasSprite sprite = GuiSpriteManager.getInstance()
                .getSprite(GUI_TEXTURE_ID_CONVERTER.fileToId(location));

        // check if the atlas doesn't have this sprite, default to using the resloc as is if so
        if (!sprite.contents().name().equals(MissingTextureAtlasSprite.getLocation())) {
            RenderSystem.setShaderTexture(0, sprite.atlasLocation());

            // have to multiply by 16 here because of MC weirdness
            // REMOVE THE MULTIPLICATION IN 1.21!!!
            return new RectangleF(sprite.getU(u0 * 16), sprite.getV(u0 * 16), sprite.getU(u1 * 16), sprite.getV(v1 * 16));
        } else {
            ModularUI.LOGGER.warn("Could not find texture {} in GUI atlas", location);
            RenderSystem.setShaderTexture(0, location);
            return new RectangleF(u0, v0, u1, v1);
        }
    }

    public static boolean isGuiAtlasSprite(ResourceLocation location) {
        location = GUI_TEXTURE_ID_CONVERTER.fileToId(location);
        return GuiSpriteManager.getInstance().getSprite(location).atlasLocation() != MissingTextureAtlasSprite.getLocation();
    }

    public static void drawTexture(Matrix4f pose, ResourceLocation location, float x, float y, float w, float h,
                                   int u, int v, int textureWidth, int textureHeight) {
        if (!isGuiAtlasSprite(location)) {
            RenderSystem.setShaderTexture(0, location);
            drawTexture(pose, x, y, u, v, w, h, textureWidth, textureHeight);
            return;
        }

        float tw = 1F / textureWidth;
        float th = 1F / textureHeight;
        RectangleF newUvs = setupTexture(location, u * tw, v * th, (u + w) * tw, (v + h) * th);

        drawTexture(pose, x, y, x + w, y + h, newUvs.u0(), newUvs.v0(), newUvs.u1(), newUvs.v1());
    }

    public static void drawTexture(Matrix4f pose, float x, float y, int u, int v, float w, float h,
                                   int textureW, int textureH) {
        drawTexture(pose, x, y, u, v, w, h, textureW, textureH, 0);
    }

    /**
     * Draw a textured quad with given UV, dimensions and custom texture size
     */
    public static void drawTexture(Matrix4f pose, float x, float y, int u, int v, float w, float h,
                                   int textureW, int textureH, float z) {
        Tesselator tesselator = Tesselator.getInstance();
        BufferBuilder buffer = tesselator.getBuilder();

        buffer.begin(VertexFormat.Mode.QUADS, DefaultVertexFormat.POSITION_TEX);
        drawTexture(pose, buffer, x, y, u, v, w, h, textureW, textureH, z);
        tesselator.end();
    }

    public static void drawTexture(Matrix4f pose, VertexConsumer buffer, float x, float y, int u, int v,
                                   float w, float h, int textureW, int textureH, float z) {
        float tw = 1F / textureW;
        float th = 1F / textureH;

        buffer.vertex(pose, x, y + h, z).uv(u * tw, (v + h) * th).endVertex();
        buffer.vertex(pose, x + w, y + h, z).uv((u + w) * tw, (v + h) * th).endVertex();
        buffer.vertex(pose, x + w, y, z).uv((u + w) * tw, v * th).endVertex();
        buffer.vertex(pose, x, y, z).uv(u * tw, v * th).endVertex();
    }

    public static void drawTexture(Matrix4f pose, float x, float y, int u, int v, float w, float h,
                                   int textureW, int textureH, int tu, int tv) {
        drawTexture(pose, x, y, u, v, w, h, textureW, textureH, tu, tv, 0);
    }

    /**
     * Draw a textured quad with given UV, dimensions and custom texture size
     */
    public static void drawTexture(Matrix4f pose, float x, float y, int u, int v, float w, float h,
                                   int textureW, int textureH, int tu, int tv, float z) {
        Tesselator tesselator = Tesselator.getInstance();
        BufferBuilder buffer = tesselator.getBuilder();

        buffer.begin(VertexFormat.Mode.QUADS, DefaultVertexFormat.POSITION_TEX);
        drawTexture(pose, buffer, x, y, u, v, w, h, textureW, textureH, tu, tv, z);
        tesselator.end();
    }

    public static void drawTexture(Matrix4f pose, VertexConsumer buffer, float x, float y, int u, int v,
                                   float w, float h, int textureW, int textureH, int tu, int tv, float z) {
        float tw = 1F / textureW;
        float th = 1F / textureH;

        buffer.vertex(pose, x, y + h, z).uv(u * tw, tv * th).endVertex();
        buffer.vertex(pose, x + w, y + h, z).uv(tu * tw, tv * th).endVertex();
        buffer.vertex(pose, x + w, y, z).uv(tu * tw, v * th).endVertex();
        buffer.vertex(pose, x, y, z).uv(u * tw, v * th).endVertex();
    }

    public static void drawTexture(Matrix4f pose, ResourceLocation location, float x0, float y0, float x1, float y1,
                                   float u0, float v0, float u1, float v1) {
        drawTexture(pose, location, x0, y0, x1, y1, u0, v0, u1, v1, false);
    }

    public static void drawTexture(Matrix4f pose, ResourceLocation location, float x0, float y0, float x1, float y1,
                                   float u0, float v0, float u1, float v1, boolean withBlend) {
        RectangleF newUvs = setupTexture(location, u0, v0, u1, v1);

        RenderSystem.setShader(GameRenderer::getPositionTexShader);
        if (withBlend) {
            RenderSystem.enableBlend();
        } else {
            RenderSystem.disableBlend();
        }
        drawTexture(pose, x0, y0, x1, y1, newUvs.u0(), newUvs.v0(), newUvs.u1(), newUvs.v1(), 0);
    }

    public static void drawTexture(Matrix4f pose, float x0, float y0, float x1, float y1,
                                   float u0, float v0, float u1, float v1) {
        drawTexture(pose, x0, y0, x1, y1, u0, v0, u1, v1, 0);
    }

    public static void drawTexture(Matrix4f pose, float x0, float y0, float x1, float y1,
                                   float u0, float v0, float u1, float v1, float z) {
        RenderSystem.disableDepthTest();
        Tesselator tesselator = Tesselator.getInstance();
        BufferBuilder buffer = tesselator.getBuilder();
        buffer.begin(VertexFormat.Mode.QUADS, DefaultVertexFormat.POSITION_TEX);
        drawTexture(pose, buffer, x0, y0, x1, y1, u0, v0, u1, v1, z);
        tesselator.end();
    }

    public static void drawTexture(Matrix4f pose, VertexConsumer buffer, float x0, float y0, float x1, float y1,
                                   float u0, float v0, float u1, float v1, float z) {
        buffer.vertex(pose, x0, y1, z).uv(u0, v1).endVertex();
        buffer.vertex(pose, x1, y1, z).uv(u1, v1).endVertex();
        buffer.vertex(pose, x1, y0, z).uv(u1, v0).endVertex();
        buffer.vertex(pose, x0, y0, z).uv(u0, v0).endVertex();
    }

    public static void drawTiledTexture(Matrix4f pose, ResourceLocation location, float x, float y, float w, float h,
                                        int u, int v, int tileWidth, int tileHeight, int textureWidth, int textureHeight, float z) {
        if (!isGuiAtlasSprite(location)) {
            RenderSystem.setShaderTexture(0, location);
            drawTiledTexture(pose, x, y, w, h, u, v, tileWidth, tileHeight, textureWidth, textureHeight, z);
            return;
        }

        float wRatio = 1f / textureWidth;
        float hRatio = 1f / textureHeight;
        RectangleF newUvs = setupTexture(location, u * wRatio, v * hRatio, (u + w) * wRatio, (v + h) * hRatio);

        drawTiledTexture(pose, x, y, x + w, y + h, newUvs.u0(), newUvs.v0(),
                newUvs.u1(), newUvs.v1(), tileWidth, tileHeight, z);
    }

    public static void drawTiledTexture(Matrix4f pose, float x, float y, float w, float h,
                                        int u, int v, int tileW, int tileH, int tw, int th, float z) {
        int countX = (((int) w - 1) / tileW) + 1;
        int countY = (((int) h - 1) / tileH) + 1;
        float fillerX = w - (countX - 1) * tileW;
        float fillerY = h - (countY - 1) * tileH;

        RenderSystem.disableDepthTest();
        Tesselator tesselator = Tesselator.getInstance();
        BufferBuilder buffer = tesselator.getBuilder();

        buffer.begin(VertexFormat.Mode.QUADS, DefaultVertexFormat.POSITION_TEX);

        for (int i = 0, c = countX * countY; i < c; i++) {
            int ix = i % countX;
            int iy = i / countX;
            float xx = x + ix * tileW;
            float yy = y + iy * tileH;
            float xw = ix == countX - 1 ? fillerX : tileW;
            float yh = iy == countY - 1 ? fillerY : tileH;

            drawTexture(pose, buffer, xx, yy, u, v, xw, yh, tw, th, z);
        }
    }

    public static void drawTiledTexture(Matrix4f pose, ResourceLocation location, float x, float y, float w, float h,
                                        float u0, float v0, float u1, float v1,
                                        int textureWidth, int textureHeight, float z) {
        RenderSystem.enableBlend();

        RectangleF newUvs = setupTexture(location, u0, v0, u1, v1);

        drawTiledTexture(pose, x, y, w, h, newUvs.u0(), newUvs.v0(), newUvs.u1(), newUvs.v1(), textureWidth, textureHeight, z);
        RenderSystem.disableBlend();
    }

    public static void drawTiledTexture(Matrix4f pose, float x, float y, float w, float h,
                                        float u0, float v0, float u1, float v1,
                                        int tileWidth, int tileHeight, float z) {
        int countX = (((int) w - 1) / tileWidth) + 1;
        int countY = (((int) h - 1) / tileHeight) + 1;
        float fillerX = w - (countX - 1) * tileWidth;
        float fillerY = h - (countY - 1) * tileHeight;
        float fillerU = u0 + (u1 - u0) * fillerX / tileWidth;
        float fillerV = v0 + (v1 - v0) * fillerY / tileHeight;

        RenderSystem.disableDepthTest();
        Tesselator tesselator = Tesselator.getInstance();
        BufferBuilder buffer = tesselator.getBuilder();

        buffer.begin(VertexFormat.Mode.QUADS, DefaultVertexFormat.POSITION_TEX);

        for (int i = 0, c = countX * countY; i < c; i++) {
            int ix = i % countX;
            int iy = i / countX;
            float xx = x + ix * tileWidth;
            float yy = y + iy * tileHeight;
            float xw = tileWidth, yh = tileHeight, uEnd = u1, vEnd = v1;
            if (ix == countX - 1) {
                xw = fillerX;
                uEnd = fillerU;
            }
            if (iy == countY - 1) {
                yh = fillerY;
                vEnd = fillerV;
            }

            drawTexture(pose, buffer, xx, yy, xx + xw, yy + yh, u0, v0, uEnd, vEnd, z);
        }

        tesselator.end();
    }

    /**
     * Draws an entity. Note that this does NOT do any necessary setup for rendering the entity. Please see
     * {@link #drawEntity(GuiGraphics, Entity, float, float, float, float, float, BiConsumer, BiConsumer)} for a full
     * draw method.
     *
     * @param graphics the current {@link GuiGraphics} instance.
     * @param entity   entity to draw.
     * @see #drawEntity(GuiGraphics, Entity, float, float, float, float, float, BiConsumer, BiConsumer)
     */
    public static void drawEntityRaw(GuiGraphics graphics, Entity entity) {
        EntityRenderDispatcher entityRenderer = Minecraft.getInstance().getEntityRenderDispatcher();
        entityRenderer.setRenderShadow(false);

        RenderSystem.runAsFancy(() -> {
            entityRenderer.render(entity, 0.0D, 0.0D, 0.0D, 0.0f, Minecraft.getInstance().getPartialTick(),
                    graphics.pose(), graphics.bufferSource(), LightTexture.FULL_BRIGHT);
        });
        graphics.flush();
        entityRenderer.setRenderShadow(true);
    }

    /**
     * A simple method to a draw an entity in a GUI. Using the consumers is not always ideal to modify and restore
     * entity state. In those cases just copy and paste this method and put your code where the consumers would be
     * called. The entity will be scaled so that it fits right in the given size when untransformed (default). When
     * transforming during pre draw, you may need to manually correct the scale and offset.
     *
     * @param graphics the current {@link GuiGraphics} instance.
     * @param entity   entity to draw
     * @param x        x pos
     * @param y        y pos
     * @param w        the width of the area where the entity should be drawn
     * @param h        the height of the area where the entity should be drawn
     * @param z        the z layer ({@link GuiContext#getCurrentDrawingZ()} if drawn in a MUI screen)
     * @param preDraw  a function to call before rendering. Transform or modify the entity here.
     * @param postDraw a function to call after rendering. Restore old entity state here if needed.
     * @param <T>      type of the entity to render
     */
    public static <T extends Entity> void drawEntity(GuiGraphics graphics, T entity,
                                                     float x, float y, float w, float h, float z,
                                                     @Nullable BiConsumer<GuiGraphics, T> preDraw,
                                                     @Nullable BiConsumer<GuiGraphics, T> postDraw) {
        graphics.pose().pushPose();
        setupDrawEntity(graphics, entity, x, y, w, h, z);
        if (preDraw != null) preDraw.accept(graphics, entity);
        drawEntityRaw(graphics, entity);
        if (postDraw != null) postDraw.accept(graphics, entity);
        endDrawEntity();
        graphics.pose().popPose();
    }

    /**
     * Draws an entity which looks in the direction of the mouse like the player render in the player inventory does.
     * The code was copied from
     * {@link net.minecraft.client.gui.screens.inventory.InventoryScreen#renderEntityInInventoryFollowsMouse(GuiGraphics, int, int, int, float, float, LivingEntity)
     * InventoryScreen.renderEntityInInventoryFollowsMouse}.
     *
     * @param graphics the current {@link GuiGraphics} instance.
     * @param entity   entity to draw
     * @param x        x pos
     * @param y        y pos
     * @param w        the width of the area where the entity should be drawn
     * @param h        the height of the area where the entity should be drawn
     * @param z        the z layer ({@link GuiContext#getCurrentDrawingZ()} if drawn in a MUI screen)
     * @param mouseX   current x pos of the mouse
     * @param mouseY   current y pos of the mouse
     */
    public static <T extends Entity> void drawEntityLookingAtMouse(GuiGraphics graphics, T entity,
                                                                   float x, float y, float w, float h, float z,
                                                                   int mouseX, int mouseY,
                                                                   @Nullable BiConsumer<GuiGraphics, T> preDraw,
                                                                   @Nullable BiConsumer<GuiGraphics, T> postDraw) {
        float xAngle = (float) Math.atan((x + w / 2 - mouseX) / h);
        float yAngle = (float) Math.atan((y + h / 2 - mouseY) / h);
        drawEntityLookingAtAngle(graphics, entity, x, y, w, h, z, xAngle, yAngle, preDraw, postDraw);
    }

    /**
     * Draws an entity which looks toward a specific angle.
     * The code was copied from
     * {@link net.minecraft.client.gui.screens.inventory.InventoryScreen#renderEntityInInventoryFollowsAngle(GuiGraphics, int, int, int, float, float, LivingEntity)
     * InventoryScreen.renderEntityInInventoryFollowsAngle}.
     *
     * @param graphics the current {@link GuiGraphics} instance.
     * @param entity   entity to draw
     * @param x        x pos
     * @param y        y pos
     * @param w        the width of the area where the entity should be drawn
     * @param h        the height of the area where the entity should be drawn
     * @param z        the z layer ({@link GuiContext#getCurrentDrawingZ()} if drawn in a MUI screen)
     * @param xAngle   the x angle to look toward
     * @param yAngle   the y angle to look toward
     */
    public static <T extends Entity> void drawEntityLookingAtAngle(GuiGraphics graphics, T entity,
                                                                   float x, float y, float w, float h, float z,
                                                                   float xAngle, float yAngle,
                                                                   @Nullable BiConsumer<GuiGraphics, T> preDraw,
                                                                   @Nullable BiConsumer<GuiGraphics, T> postDraw) {
        graphics.pose().pushPose();
        setupDrawEntity(graphics, entity, x, y, w, h, z);

        // pre draw
        float oldYRot = entity.getYRot();
        float oldYRotO = entity.yRotO;
        float oldXRot = entity.getXRot();
        float oldXRotO = entity.xRotO;
        float oldYBodyRot = 0.0f;
        float oldYBodyRotO = 0.0f;
        float oldYHeadRotO = 0.0f;
        float oldYHeadRot = 0.0f;

        entity.setYRot(entity.yRotO = 180.0f + xAngle * 40.0f);
        entity.setXRot(entity.xRotO = -yAngle * 20.0f);
        // made this method more generic by only updating these if the entity is a LivingEntity
        if (entity instanceof LivingEntity livingEntity) {
            oldYBodyRot = livingEntity.yBodyRot;
            oldYBodyRotO = livingEntity.yBodyRotO;
            oldYHeadRotO = livingEntity.yHeadRotO;
            oldYHeadRot = livingEntity.yHeadRot;

            livingEntity.yBodyRotO = livingEntity.yBodyRot = 180.0f + xAngle * 20.0f;
            livingEntity.yHeadRotO = livingEntity.yHeadRot = entity.getYRot();
        }

        // skip rotating the render by 180° on the Z axis here, because we always do that in setupDrawEntity
        Quaternionf cameraRot = new Quaternionf().rotateX(yAngle * 20.0f * Mth.DEG_TO_RAD);
        graphics.pose().mulPose(cameraRot);
        // set the camera orientation (vanilla also does this)
        cameraRot.conjugate();
        Minecraft.getInstance().getEntityRenderDispatcher().overrideCameraOrientation(cameraRot);

        if (preDraw != null) preDraw.accept(graphics, entity);
        drawEntityRaw(graphics, entity);
        if (postDraw != null) postDraw.accept(graphics, entity);

        // post draw
        entity.setYRot(oldYRot);
        entity.yRotO = oldYRotO;
        entity.setXRot(oldXRot);
        entity.xRotO = oldXRotO;
        if (entity instanceof LivingEntity livingEntity) {
            livingEntity.yBodyRot = oldYBodyRot;
            livingEntity.yBodyRotO = oldYBodyRotO;
            livingEntity.yHeadRotO = oldYHeadRotO;
            livingEntity.yHeadRot = oldYHeadRot;
        }

        endDrawEntity();
        graphics.pose().popPose();
    }

    /**
     * Sets up the gl state for rendering an entity. The entity will be scaled so that it fits right in the given size
     * when untransformed.
     *
     * @param graphics the current {@link GuiGraphics} instance.
     * @param entity   entity to set up drawing for
     * @param x        x pos
     * @param y        y pos
     * @param w        the width of the area where the entity should be drawn
     * @param h        the height of the area where the entity should be drawn
     * @param z        the z layer ({@link GuiContext#getCurrentDrawingZ()} if drawn in a MUI)
     */
    public static void setupDrawEntity(GuiGraphics graphics, Entity entity, float x, float y,
                                       float w, float h, float z) {
        float size;
        float scale;
        if (h / entity.getBbHeight() < w / entity.getBbWidth()) {
            size = entity.getBbHeight();
            scale = h / size;
        } else {
            size = entity.getBbWidth();
            scale = w / size;
        }
        graphics.pose().translate(x + w / 2, y + h / 2, z + 50.0f);
        graphics.pose().scale(scale, scale, -scale);
        graphics.pose().translate(0, size / 2f, 0);
        graphics.pose().mulPose(new Quaternionf().rotateZ(Mth.PI));

        Lighting.setupForEntityInInventory();
    }

    public static void endDrawEntity() {
        Lighting.setupFor3DItems();
    }

    public static void drawItem(GuiGraphics graphics, ItemStack item, int x, int y, float width, float height, int z) {
        if (item.isEmpty()) return;
        graphics.pose().pushPose();
        graphics.pose().translate(x, y, z);
        graphics.pose().scale(width / 16f, height / 16f, 1);
        graphics.renderItem(item, 0, 0);
        graphics.renderItemDecorations(Minecraft.getInstance().font, item, 0, 0);
        graphics.pose().popPose();
    }

    public static void drawFluidTexture(GuiGraphics graphics, FluidStack content, float x0, float y0,
                                        float width, float height, float z) {
        if (content == null || content.isEmpty()) {
            return;
        }
        Fluid fluid = content.getFluid();
        ResourceLocation fluidStill = IClientFluidTypeExtensions.of(fluid).getStillTexture(content);
        TextureAtlasSprite sprite = Minecraft.getInstance().getTextureAtlas(InventoryMenu.BLOCK_ATLAS)
                .apply(fluidStill);
        int fluidColor = IClientFluidTypeExtensions.of(fluid).getTintColor(content);
        graphics.setColor(Color.getRedF(fluidColor), Color.getGreenF(fluidColor), Color.getBlueF(fluidColor),
                Color.getAlphaF(fluidColor));
        RenderSystem.setShader(GameRenderer::getPositionTexShader);
        drawTiledTexture(graphics.pose().last().pose(), InventoryMenu.BLOCK_ATLAS, x0, y0, width, height,
                sprite.getU0(), sprite.getV0(),
                sprite.getU1(), sprite.getV1(), sprite.contents().width(), sprite.contents().height(), z);
        graphics.setColor(1f, 1f, 1f, 1f);
    }

    public static void drawStandardSlotAmountText(GuiContext context, int amount, String format, Area area,
                                                  float z) {
        drawAmountText(context.getMuiContext(), amount, format, 0, 0, area.width, area.height, Alignment.BottomRight,
                z);
    }

    public static void drawAmountText(ModularGuiContext context, int amount, String format,
                                      int x, int y, int width, int height, Alignment alignment, float z) {
        if (amount == 1) return;

        String amountText = FormattingUtil.formatNumberReadable(amount, false);
        if (format != null) {
            amountText = format + amountText;
        }
        drawScaledAlignedTextInBox(context, amountText, x, y, width, height, alignment, 1f, z);
    }

    public static void drawScaledAlignedTextInBox(ModularGuiContext context, String amountText,
                                                  int x, int y, int width, int height,
                                                  Alignment alignment) {
        drawScaledAlignedTextInBox(context, amountText, x, y, width, height, alignment, 1f, 0.0f);
    }

    public static void drawScaledAlignedTextInBox(ModularGuiContext context, String amountText,
                                                  int x, int y, int width, int height,
                                                  Alignment alignment, float maxScale, float z) {
        if (amountText == null || amountText.isEmpty()) return;
        // render the amount overlay
        textRenderer.setShadow(true);
        textRenderer.setScale(1f);
        textRenderer.setColor(Color.WHITE.main);
        textRenderer.setAlignment(alignment, width, height);
        textRenderer.setPos(x, y);
        textRenderer.setHardWrapOnBorder(false);
        RenderSystem.disableDepthTest();
        RenderSystem.disableBlend();
        if (amountText.length() > 2 && width > 16) { // we know that numbers below 100 will always fit in standard slots
            // simulate and calculate scale with width
            textRenderer.setSimulate(true);
            textRenderer.draw(context.getGraphics(), amountText);
            textRenderer.setSimulate(false);
            textRenderer.setScale(Math.min(maxScale, width / textRenderer.getLastWidth()));
        }
        context.graphicsPose().translate(0, 0, 100 + z);
        textRenderer.draw(context.getGraphics(), amountText);
        textRenderer.setHardWrapOnBorder(true);
        RenderSystem.enableDepthTest();
        RenderSystem.enableBlend();
    }

    public static void drawSprite(Matrix4f pose, TextureAtlasSprite sprite, float x0, float y0, float w, float h) {
        RenderSystem.enableBlend();
        RenderSystem.setShaderTexture(0, sprite.atlasLocation());
        drawTexture(pose, x0, y0, x0 + w, y0 + h, sprite.getU0(), sprite.getV0(), sprite.getU1(), sprite.getV1());
        RenderSystem.disableBlend();
    }

    public static void drawTiledSprite(Matrix4f pose, TextureAtlasSprite sprite, float x0, float y0, float w, float h) {
        RenderSystem.enableBlend();
        RenderSystem.setShaderTexture(0, sprite.atlasLocation());
        drawTiledTexture(pose, sprite.atlasLocation(), x0, y0, x0 + w, y0 + h, sprite.getU0(), sprite.getV0(),
                sprite.getU1(), sprite.getV1(), sprite.contents().width(), sprite.contents().height(), 0);
        RenderSystem.disableBlend();
    }

    public static void drawOutlineCenter(GuiGraphics graphics, int x, int y, int offset, int color) {
        drawOutlineCenter(graphics, x, y, offset, color, 1);
    }

    public static void drawOutlineCenter(GuiGraphics graphics, int x, int y, int offset, int color, int border) {
        drawOutline(graphics, x - offset, y - offset, x + offset, y + offset, color, border);
    }

    public static void drawOutline(GuiGraphics graphics, int left, int top, int right, int bottom, int color) {
        drawOutline(graphics, left, top, right, bottom, color, 1);
    }

    /**
     * Draw rectangle outline with given border
     */
    public static void drawOutline(GuiGraphics graphics, int left, int top, int right, int bottom,
                                   int color, int border) {
        graphics.fill(left, top, left + border, bottom, color);
        graphics.fill(right - border, top, right, bottom, color);
        graphics.fill(left + border, top, right - border, top + border, color);
        graphics.fill(left + border, bottom - border, right - border, bottom, color);
    }

    private static void drawBorderLTRB(GuiGraphics graphics, float left, float top, float right, float bottom,
                                       float border, int color, boolean outside) {
        if (outside) {
            left -= border;
            top -= border;
            right += border;
            bottom += border;
        }
        float x0 = left, y0 = top, x1 = right, y1 = bottom, d = border;

        var buffer = graphics.bufferSource().getBuffer(GTRenderTypes.guiTriangleStrip());
        var pose = graphics.pose().last().pose();
        pc(buffer, pose, x0, y0, color);
        pc(buffer, pose, x1 - d, y0 + d, color);
        pc(buffer, pose, x1, y0, color);
        pc(buffer, pose, x1 - d, y1 - d, color);
        pc(buffer, pose, x1, y1, color);
        pc(buffer, pose, x0 + d, y1 - d, color);
        pc(buffer, pose, x0, y1, color);
        pc(buffer, pose, x0 + d, y0 + d, color);
        pc(buffer, pose, x0, y0, color);
        pc(buffer, pose, x1 - d, y0 + d, color);
    }

    public static void drawBorderOutsideLTRB(GuiGraphics graphics, float left, float top, float right, float bottom,
                                             int color) {
        drawBorderLTRB(graphics, left, top, right, bottom, 1, color, true);
    }

    public static void drawBorderOutsideLTRB(GuiGraphics graphics, float left, float top, float right, float bottom,
                                             float border, int color) {
        drawBorderLTRB(graphics, left, top, right, bottom, border, color, true);
    }

    public static void drawBorderInsideLTRB(GuiGraphics graphics, float left, float top, float right, float bottom,
                                            int color) {
        drawBorderLTRB(graphics, left, top, right, bottom, 1, color, false);
    }

    public static void drawBorderInsideLTRB(GuiGraphics graphics, float left, float top, float right, float bottom,
                                            float border, int color) {
        drawBorderLTRB(graphics, left, top, right, bottom, border, color, false);
    }

    private static void drawBorderXYWH(GuiGraphics graphics, float x, float y, float w, float h, float border,
                                       int color, boolean outside) {
        drawBorderLTRB(graphics, x, y, x + w, y + h, border, color, outside);
    }

    public static void drawBorderOutsideXYWH(GuiGraphics graphics, float x, float y, float w, float h, float border,
                                             int color) {
        drawBorderXYWH(graphics, x, y, w, h, border, color, true);
    }

    public static void drawBorderOutsideXYWH(GuiGraphics graphics, float x, float y, float w, float h, int color) {
        drawBorderXYWH(graphics, x, y, w, h, 1, color, true);
    }

    public static void drawBorderInsideXYWH(GuiGraphics graphics, float x, float y, float w, float h, float border,
                                            int color) {
        drawBorderXYWH(graphics, x, y, w, h, border, color, false);
    }

    public static void drawBorderInsideXYWH(GuiGraphics graphics, float x, float y, float w, float h, int color) {
        drawBorderXYWH(graphics, x, y, w, h, 1, color, false);
    }

    private static void pc(VertexConsumer buffer, Matrix4f pose, float x, float y, int c) {
        buffer.vertex(pose, x, y, 0).color(Color.getRed(c), Color.getGreen(c), Color.getBlue(c), Color.getAlpha(c))
                .endVertex();
    }

    /**
     * Draws a rectangular shadow
     *
     * @param x      left of solid shadow part
     * @param y      top of solid shadow part
     * @param w      width of solid shadow part
     * @param h      height of solid shadow part
     * @param oX     shadow gradient size in x
     * @param oY     shadow gradient size in y
     * @param opaque solid shadow color
     * @param shadow gradient end color
     */
    public static void drawDropShadow(Matrix4f pose, int x, int y, int w, int h, int oX, int oY,
                                      int opaque, int shadow) {
        float a1 = Color.getAlphaF(opaque);
        float r1 = Color.getRedF(opaque);
        float g1 = Color.getGreenF(opaque);
        float b1 = Color.getBlueF(opaque);
        float a2 = Color.getAlphaF(shadow);
        float r2 = Color.getRedF(shadow);
        float g2 = Color.getGreenF(shadow);
        float b2 = Color.getBlueF(shadow);

        RenderSystem.setShader(GameRenderer::getPositionColorShader);
        RenderSystem.enableBlend();
        RenderSystem.disableDepthTest();
        RenderSystem.blendFuncSeparate(GlStateManager.SourceFactor.SRC_ALPHA,
                GlStateManager.DestFactor.ONE_MINUS_SRC_ALPHA,
                GlStateManager.SourceFactor.ONE, GlStateManager.DestFactor.ZERO);

        Tesselator tesselator = Tesselator.getInstance();
        BufferBuilder buffer = tesselator.getBuilder();

        buffer.begin(VertexFormat.Mode.QUADS, DefaultVertexFormat.POSITION_COLOR);

        float x1 = x + w, y1 = y + h;

        /* Draw opaque part */
        buffer.vertex(pose, x1, y, 0).color(r1, g1, b1, a1).endVertex();
        buffer.vertex(pose, x, y, 0).color(r1, g1, b1, a1).endVertex();
        buffer.vertex(pose, x, y1, 0).color(r1, g1, b1, a1).endVertex();
        buffer.vertex(pose, x1, y1, 0).color(r1, g1, b1, a1).endVertex();

        /* Draw top shadow */
        buffer.vertex(pose, x1 + oX, y - oY, 0).color(r2, g2, b2, a2).endVertex();
        buffer.vertex(pose, x - oX, y - oY, 0).color(r2, g2, b2, a2).endVertex();
        buffer.vertex(pose, x, y, 0).color(r1, g1, b1, a1).endVertex();
        buffer.vertex(pose, x1, y, 0).color(r1, g1, b1, a1).endVertex();

        /* Draw bottom shadow */
        buffer.vertex(pose, x1, y1, 0).color(r1, g1, b1, a1).endVertex();
        buffer.vertex(pose, x, y1, 0).color(r1, g1, b1, a1).endVertex();
        buffer.vertex(pose, x - oX, y1 + oY, 0).color(r2, g2, b2, a2).endVertex();
        buffer.vertex(pose, x1 + oX, y1 + oY, 0).color(r2, g2, b2, a2).endVertex();

        /* Draw left shadow */
        buffer.vertex(pose, x, y, 0).color(r1, g1, b1, a1).endVertex();
        buffer.vertex(pose, x - oX, y - oY, 0).color(r2, g2, b2, a2).endVertex();
        buffer.vertex(pose, x - oX, y1 + oY, 0).color(r2, g2, b2, a2).endVertex();
        buffer.vertex(pose, x, y1, 0).color(r1, g1, b1, a1).endVertex();

        /* Draw right shadow */
        buffer.vertex(pose, x1 + oX, y - oY, 0).color(r2, g2, b2, a2).endVertex();
        buffer.vertex(pose, x1, y, 0).color(r1, g1, b1, a1).endVertex();
        buffer.vertex(pose, x1, y1, 0).color(r1, g1, b1, a1).endVertex();
        buffer.vertex(pose, x1 + oX, y1 + oY, 0).color(r2, g2, b2, a2).endVertex();

        tesselator.end();
        RenderSystem.disableBlend();
    }

    public static void drawDropCircleShadow(GuiGraphics graphics, int x, int y, int radius, int segments,
                                            int opaque, int shadow) {
        Matrix4f pose = graphics.pose().last().pose();
        Matrix4d poseD = new Matrix4d(pose);

        float a1 = Color.getAlphaF(opaque);
        float r1 = Color.getRedF(opaque);
        float g1 = Color.getGreenF(opaque);
        float b1 = Color.getBlueF(opaque);
        float a2 = Color.getAlphaF(shadow);
        float r2 = Color.getRedF(shadow);
        float g2 = Color.getGreenF(shadow);
        float b2 = Color.getBlueF(shadow);

        VertexConsumer buffer = graphics.bufferSource().getBuffer(GTRenderTypes.guiOverlayTriangleFan());
        buffer.vertex(pose, x, y, 0).color(r1, g1, b1, a1).endVertex();

        Vector3d pos = new Vector3d();
        for (int i = 0; i <= segments; i++) {
            float a = i / (float) segments * TWO_PI - HALF_PI;
            circleVertex(buffer, poseD, pos, x, Mth.cos(a), y, Mth.sin(a), radius).color(r2, g2, b2, a2).endVertex();
        }
    }

    public static void drawDropCircleShadow(GuiGraphics graphics, int x, int y, int radius, int offset, int segments,
                                            int opaque, int shadow) {
        if (offset >= radius) {
            drawDropCircleShadow(graphics, x, y, radius, segments, opaque, shadow);
            return;
        }
        Matrix4f pose = graphics.pose().last().pose();
        Matrix4d poseD = new Matrix4d(pose);

        float a1 = Color.getAlphaF(opaque);
        float r1 = Color.getRedF(opaque);
        float g1 = Color.getGreenF(opaque);
        float b1 = Color.getBlueF(opaque);
        float a2 = Color.getAlphaF(shadow);
        float r2 = Color.getRedF(shadow);
        float g2 = Color.getGreenF(shadow);
        float b2 = Color.getBlueF(shadow);

        VertexConsumer buffer = graphics.bufferSource().getBuffer(GTRenderTypes.guiOverlayTriangleFan());
        /* Draw opaque base */
        buffer.vertex(pose, x, y, 0).color(r1, g1, b1, a1).endVertex();

        Vector3d pos = new Vector3d();
        for (int i = 0; i <= segments; i++) {
            float a = i / (float) segments * TWO_PI - HALF_PI;
            circleVertex(buffer, poseD, pos, x, Mth.cos(a), y, Mth.sin(a), offset).color(r1, g1, b1, a1).endVertex();
        }

        /* Draw outer shadow */
        buffer = graphics.bufferSource().getBuffer(RenderType.gui());

        for (int i = 0; i < segments; i++) {
            float alpha1 = i / (float) segments * TWO_PI - HALF_PI;
            float alpha2 = (i + 1) / (float) segments * TWO_PI - HALF_PI;

            float cosA1 = Mth.cos(alpha1);
            float cosA2 = Mth.cos(alpha2);
            float sinA1 = Mth.sin(alpha1);
            float sinA2 = Mth.sin(alpha2);

            circleVertex(buffer, poseD, pos, x, cosA2, y, sinA2, offset).color(r1, g1, b1, a1).endVertex();
            circleVertex(buffer, poseD, pos, x, cosA1, y, sinA1, offset).color(r1, g1, b1, a1).endVertex();
            circleVertex(buffer, poseD, pos, x, cosA1, y, sinA1, radius).color(r2, g2, b2, a2).endVertex();
            circleVertex(buffer, poseD, pos, x, cosA2, y, sinA2, radius).color(r2, g2, b2, a2).endVertex();
        }
    }

    private static VertexConsumer circleVertex(VertexConsumer buffer, Matrix4d pose, Vector3d pos,
                                               float x, float xOffset, float y, float yOffset, float mul) {
        pos.x = x - xOffset * mul;
        pos.y = y + yOffset * mul;
        pose.transformPosition(pos);
        return buffer.vertex(pos.x, pos.y, pos.z);
    }

    @OnlyIn(Dist.CLIENT)
    public static void drawBorder(GuiGraphics graphics, float x, float y, float width, float height,
                                  int color, float border) {
        drawBorderLTRB(graphics, x, y, x + width, y + height, border, color, false);
        // drawRect(graphics, x - border, y - border, width + 2 * border, border, color);
        // drawRect(graphics, x - border, y + height, width + 2 * border, border, color);
        // drawRect(graphics, x - border, y, border, height, color);
        // drawRect(graphics, x + width, y, border, height, color);
    }

    @OnlyIn(Dist.CLIENT)
    public static void drawText(GuiGraphics graphics, String text, float x, float y, float scale,
                                int color, boolean shadow) {
        graphics.pose().pushPose();
        graphics.pose().scale(scale, scale, 0f);
        float sf = 1 / scale;
        graphics.drawString(Minecraft.getInstance().font, text, x * sf, y * sf, color, shadow);
        graphics.pose().popPose();
    }

    @OnlyIn(Dist.CLIENT)
    public static void drawText(GuiGraphics graphics, Component text, float x, float y, float scale,
                                int color, boolean shadow) {
        drawText(graphics, text.getVisualOrderText(), x, y, scale, color, shadow);
    }

    @OnlyIn(Dist.CLIENT)
    public static void drawText(GuiGraphics graphics, FormattedCharSequence text, float x, float y, float scale,
                                int color, boolean shadow) {
        graphics.pose().pushPose();
        graphics.pose().scale(scale, scale, 0f);
        float sf = 1 / scale;
        graphics.drawString(Minecraft.getInstance().font, text, x * sf, y * sf, color, shadow);
        graphics.pose().popPose();
    }

    @SuppressWarnings("UnstableApiUsage")
    public static void drawTooltipBackground(GuiContext context, ItemStack stack, List<ClientTooltipComponent> lines,
                                             int x, int y, int textWidth, int height, @Nullable RichTooltip tooltip) {
        GuiGraphics graphics = context.getGraphics();

        // TODO theme color
        int backgroundTop = 0xF0100010;
        int backgroundBottom = backgroundTop;
        int borderColorStart = 0x505000FF;
        int borderColorEnd = (borderColorStart & 0xFEFEFE) >> 1 | borderColorStart & 0xFF000000;
        RenderTooltipEvent.Color colorEvent;

        if (tooltip != null) {
            colorEvent = new RichTooltipEvent.Color(stack, graphics, x, y, context.getFont(),
                    backgroundTop, borderColorStart, borderColorEnd, lines, tooltip);
        } else {
            colorEvent = new RenderTooltipEvent.Color(stack, graphics, x, y, context.getFont(),
                    backgroundTop, borderColorStart, borderColorEnd, lines);
        }

        MinecraftForge.EVENT_BUS.post(colorEvent);
        backgroundTop = colorEvent.getBackgroundStart();
        backgroundBottom = colorEvent.getBackgroundEnd();
        borderColorStart = colorEvent.getBorderStart();
        borderColorEnd = colorEvent.getBorderEnd();

        // top background border
        drawVerticalGradientRect(graphics, x - 3, y - 4, textWidth + 6, 1, backgroundTop, backgroundTop);
        // bottom background border
        drawVerticalGradientRect(graphics, x - 3, y + height + 3, textWidth + 6, 1, backgroundBottom, backgroundBottom);
        // center background
        drawVerticalGradientRect(graphics, x - 3, y - 3, textWidth + 6, height + 6, backgroundTop, backgroundBottom);
        // left background border
        drawVerticalGradientRect(graphics, x - 4, y - 3, 1, height + 6, backgroundTop, backgroundBottom);
        // right background border
        drawVerticalGradientRect(graphics, x + textWidth + 3, y - 3, 1, height + 6, backgroundTop, backgroundBottom);

        // left accent border
        drawVerticalGradientRect(graphics, x - 3, y - 2, 1, height + 4, borderColorStart, borderColorEnd);
        // right accent border
        drawVerticalGradientRect(graphics, x + textWidth + 2, y - 2, 1, height + 4, borderColorStart, borderColorEnd);
        // top accent border
        drawVerticalGradientRect(graphics, x - 3, y - 3, textWidth + 6, 1, borderColorStart, borderColorStart);
        // bottom accent border
        drawVerticalGradientRect(graphics, x - 3, y + height + 2, textWidth + 6, 1, borderColorEnd, borderColorEnd);
    }
}
