/*
 * This file is part of Applied Energistics 2.
 * Copyright (c) 2021, TeamAppliedEnergistics, All rights reserved.
 *
 * Applied Energistics 2 is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Applied Energistics 2 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Applied Energistics 2.  If not, see <http://www.gnu.org/licenses/lgpl>.
 */

package appeng.menu.me.crafting;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Objects;

import com.google.common.collect.ImmutableList;

import net.minecraft.network.FriendlyByteBuf;

import appeng.api.config.Actionable;
import appeng.api.networking.IGrid;
import appeng.api.networking.crafting.ICraftingPlan;
import appeng.api.networking.security.IActionSource;
import appeng.api.stacks.AEKey;

/**
 * A crafting plan intended to be sent to the client.
 *
 * @param usedBytes
 * @param simulation
 */
public record CraftingPlanSummary(long usedBytes, boolean simulation, List<CraftingPlanSummaryEntry> entries) {

    public void write(FriendlyByteBuf buffer) {
        buffer.writeVarLong(usedBytes);
        buffer.writeBoolean(simulation);
        buffer.writeVarInt(entries.size());
        for (CraftingPlanSummaryEntry entry : entries) {
            entry.write(buffer);
        }
    }

    public static CraftingPlanSummary read(FriendlyByteBuf buffer) {

        long bytesUsed = buffer.readVarLong();
        boolean simulation = buffer.readBoolean();
        int entryCount = buffer.readVarInt();
        ImmutableList.Builder<CraftingPlanSummaryEntry> entries = ImmutableList.builder();
        for (int i = 0; i < entryCount; i++) {
            entries.add(CraftingPlanSummaryEntry.read(buffer));
        }

        return new CraftingPlanSummary(bytesUsed, simulation, entries.build());
    }

    private static class KeyStats {
        public long stored;
        public long crafting;
    }

    /**
     * Creates a plan summary from the given planning result.
     *
     * @param grid         The grid used to determine the amount of items already stored.
     * @param actionSource The action source used to determine the amount of items already stored.
     */
    public static CraftingPlanSummary fromJob(IGrid grid, IActionSource actionSource, ICraftingPlan job) {
        var plan = new HashMap<AEKey, KeyStats>() {
            private KeyStats mapping(AEKey key) {
                Objects.requireNonNull(key, "Key may not be null");

                return computeIfAbsent(key, k -> new KeyStats());
            }
        };

        for (var used : job.usedItems()) {
            plan.mapping(used.getKey()).stored += used.getLongValue();
        }
        for (var missing : job.missingItems()) {
            plan.mapping(missing.getKey()).stored += missing.getLongValue();
        }
        for (var emitted : job.emittedItems()) {
            var entry = plan.mapping(emitted.getKey());
            entry.stored += emitted.getLongValue();
            entry.crafting += emitted.getLongValue();
        }
        for (var entry : job.patternTimes().entrySet()) {
            for (var out : entry.getKey().getOutputs()) {
                plan.mapping(out.what()).crafting += out.amount() * entry.getValue();
            }
        }

        var entries = new ArrayList<CraftingPlanSummaryEntry>();

        var storage = grid.getStorageService().getInventory();
        var crafting = grid.getCraftingService();
        var cached = grid.getStorageService().getCachedInventory();

        for (var out : plan.entrySet()) {
            long missingAmount;
            long storedAmount;
            long availableAmount;
            if (job.simulation() && !crafting.canEmitFor(out.getKey())) {
                storedAmount = storage.extract(out.getKey(), out.getValue().stored, Actionable.SIMULATE, actionSource);
                missingAmount = out.getValue().stored - storedAmount;
            } else {
                storedAmount = out.getValue().stored;
                missingAmount = 0;
            }
            availableAmount = cached.get(out.getKey());
            long craftAmount = out.getValue().crafting;
            entries.add(new CraftingPlanSummaryEntry(
                    out.getKey(),
                    missingAmount,
                    storedAmount,
                    craftAmount,
                    availableAmount));

        }

        Collections.sort(entries);

        return new CraftingPlanSummary(job.bytes(), job.simulation(), List.copyOf(entries));

    }

}
