/*
 * This file is part of Applied Energistics 2.
 * Copyright (c) 2021, TeamAppliedEnergistics, All rights reserved.
 *
 * Applied Energistics 2 is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Applied Energistics 2 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Applied Energistics 2.  If not, see <http://www.gnu.org/licenses/lgpl>.
 */

package appeng.client.gui.widgets;

import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.ParsePosition;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.OptionalInt;
import java.util.OptionalLong;
import java.util.function.Consumer;

import com.google.common.primitives.Longs;

import net.minecraft.ChatFormatting;
import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.Font;
import net.minecraft.client.gui.GuiGraphics;
import net.minecraft.client.gui.components.AbstractWidget;
import net.minecraft.client.gui.components.Button;
import net.minecraft.client.gui.screens.Screen;
import net.minecraft.client.renderer.Rect2i;
import net.minecraft.network.chat.Component;

import appeng.client.Point;
import appeng.client.gui.AEBaseScreen;
import appeng.client.gui.ICompositeWidget;
import appeng.client.gui.NumberEntryType;
import appeng.client.gui.Rects;
import appeng.client.gui.style.PaletteColor;
import appeng.client.gui.style.ScreenStyle;
import appeng.client.gui.style.WidgetStyle;
import appeng.core.localization.GuiText;
import appeng.util.NumberUtil;

/**
 * A utility widget that consists of a text-field to enter a number with attached buttons to increment/decrement the
 * number in fixed intervals.
 */
public class NumberEntryWidget implements ICompositeWidget {

    private static final long[] STEPS_1000 = new long[] { 1, 10, 100, 1000 };
    private static final long[] STEPS_64 = new long[] { 1, 16, 32, 64 };
    private final Component[] components1000;
    private final Component[] components64;
    private static final Component PLUS = Component.literal("+");
    private static final Component MINUS = Component.literal("-");
    private static final int UNIT_PADDING = 3;
    private final int errorTextColor;
    private final int normalTextColor;

    private final ConfirmableTextField textField;
    private final DecimalFormat decimalFormat;
    private NumberEntryType type;
    private List<Button> buttons;
    private long minValue;
    private long maxValue = Long.MAX_VALUE;
    private ValidationIcon validationIcon;

    // Called when the value changes
    private Runnable onChange;

    // Called when the user presses enter while there's a valid number in the field
    private Runnable onConfirm;

    private boolean hideValidationIcon;

    private Rect2i bounds = new Rect2i(0, 0, 0, 0);

    private Rect2i textFieldBounds = Rects.ZERO;
    private Point currentScreenOrigin = Point.ZERO;

    private final List<Button> amountButtons = List.of();

    public NumberEntryWidget(ScreenStyle style, NumberEntryType type) {
        this.errorTextColor = style.getColor(PaletteColor.TEXTFIELD_ERROR).toARGB();
        this.normalTextColor = style.getColor(PaletteColor.TEXTFIELD_TEXT).toARGB();

        this.type = Objects.requireNonNull(type, "type");
        this.decimalFormat = new DecimalFormat("#.######", new DecimalFormatSymbols());
        this.decimalFormat.setParseBigDecimal(true);
        this.decimalFormat.setNegativePrefix("-");

        Font font = Minecraft.getInstance().font;

        this.textField = new ConfirmableTextField(style, font, 0, 0, 0, font.lineHeight);
        this.textField.setBordered(false);
        this.textField.setMaxLength(16);
        this.textField.setTextColor(normalTextColor);
        this.textField.setVisible(true);
        this.textField.setResponder(text -> {
            validate();
            if (onChange != null) {
                this.onChange.run();
            }
        });
        this.textField.setOnConfirm(() -> {
            // Only confirm if it's actually valid
            if (this.onConfirm != null && getLongValue().isPresent()) {
                this.onConfirm.run();
            }
        });
        validate();

        components1000 = new Component[] {
                makeLabel(PLUS, 0, true),
                makeLabel(PLUS, 1, true),
                makeLabel(PLUS, 2, true),
                makeLabel(PLUS, 3, true),
                makeLabel(MINUS, 0, true),
                makeLabel(MINUS, 1, true),
                makeLabel(MINUS, 2, true),
                makeLabel(MINUS, 3, true),
        };
        components64 = new Component[] {
                makeLabel(PLUS, 0, false),
                makeLabel(PLUS, 1, false),
                makeLabel(PLUS, 2, false),
                makeLabel(PLUS, 3, false),
                makeLabel(MINUS, 0, false),
                makeLabel(MINUS, 1, false),
                makeLabel(MINUS, 2, false),
                makeLabel(MINUS, 3, false),
        };
    }

    public void setOnConfirm(Runnable callback) {
        this.onConfirm = callback;
    }

    public void setOnChange(Runnable callback) {
        this.onChange = callback;
    }

    public void setActive(boolean active) {
        this.textField.setEditable(active);
        this.buttons.forEach(b -> b.active = active);
    }

    /**
     * Sets the bounds of the text field on the screen. This may seem insane, but the text-field background is actually
     * baked into the screens background image, which necessitates setting it precisely.
     */
    public void setTextFieldBounds(Rect2i bounds) {
        this.textFieldBounds = bounds;
        this.textField.move(currentScreenOrigin.move(bounds.getX(), bounds.getY()));
        int unitWidth = 0;
        if (this.type.unit() != null) {
            unitWidth = Minecraft.getInstance().font.width(this.type.unit()) + UNIT_PADDING;
        }
        this.textField.resize(bounds.getWidth() - unitWidth, bounds.getHeight());
    }

    public void setTextFieldStyle(WidgetStyle style) {
        int left = 0;
        if (style.getLeft() != null) {
            left = style.getLeft();
        }
        int top = 0;
        if (style.getTop() != null) {
            top = style.getTop();
        }
        setTextFieldBounds(new Rect2i(
                left,
                top,
                style.getWidth(),
                style.getHeight()));
    }

    public void setMinValue(long minValue) {
        this.minValue = minValue;
        validate();
    }

    public void setMaxValue(long maxValue) {
        this.maxValue = maxValue;
        validate();
    }

    @Override
    public void setPosition(Point position) {
        bounds = new Rect2i(position.getX(), position.getY(), bounds.getWidth(), bounds.getHeight());
    }

    @Override
    public void setSize(int width, int height) {
        bounds = new Rect2i(bounds.getX(), bounds.getY(), width, height);
    }

    @Override
    public Rect2i getBounds() {
        return bounds;
    }

    @Override
    public void populateScreen(Consumer<AbstractWidget> addWidget, Rect2i bounds, AEBaseScreen<?> screen) {
        int left = bounds.getX() + this.bounds.getX();
        int top = bounds.getY() + this.bounds.getY();

        List<Button> buttons = new ArrayList<>(9);

        buttons.add(
                Button.builder(components1000[0], btn -> addQty(hasShiftOrControlDown() ? STEPS_64[0] : STEPS_1000[0]))
                        .bounds(left, top, 22, 20)
                        .build());
        buttons.add(
                Button.builder(components1000[1], btn -> addQty(hasShiftOrControlDown() ? STEPS_64[1] : STEPS_1000[1]))
                        .bounds(left, top, 22, 20)
                        .build());
        buttons.add(
                Button.builder(components1000[2], btn -> addQty(hasShiftOrControlDown() ? STEPS_64[2] : STEPS_1000[2]))
                        .bounds(left, top, 22, 20)
                        .build());
        buttons.add(
                Button.builder(components1000[3], btn -> addQty(hasShiftOrControlDown() ? STEPS_64[3] : STEPS_1000[3]))
                        .bounds(left, top, 22, 20)
                        .build());

        // Need to add these now for sensible tab-order
        buttons.forEach(addWidget);

        // Placing this here will give a sensible tab order
        this.currentScreenOrigin = Point.fromTopLeft(bounds);
        setTextFieldBounds(this.textFieldBounds);
        screen.setInitialFocus(this.textField);
        addWidget.accept(this.textField);

        buttons.add(Button
                .builder(components1000[4], btn -> addQty(hasShiftOrControlDown() ? -STEPS_64[0] : -STEPS_1000[0]))
                .bounds(left, top, 22, 20)
                .build());
        buttons.add(Button
                .builder(components1000[5], btn -> addQty(hasShiftOrControlDown() ? -STEPS_64[1] : -STEPS_1000[1]))
                .bounds(left, top, 22, 20)
                .build());
        buttons.add(Button
                .builder(components1000[6], btn -> addQty(hasShiftOrControlDown() ? -STEPS_64[2] : -STEPS_1000[2]))
                .bounds(left, top, 22, 20)
                .build());
        buttons.add(Button
                .builder(components1000[7], btn -> addQty(hasShiftOrControlDown() ? -STEPS_64[3] : -STEPS_1000[3]))
                .bounds(left, top, 22, 20)
                .build());

        // This element is not focusable
        if (!hideValidationIcon) {
            this.validationIcon = new ValidationIcon();
            this.validationIcon.setX(left + 104);
            this.validationIcon.setY(top + 27);
            buttons.add(this.validationIcon);
        }

        // Add the rest to the tab order
        buttons.subList(4, buttons.size()).forEach(addWidget);

        this.buttons = buttons;

        // we need to re-validate because the icon may now be present and needs it's
        // initial state
        this.validate();
    }

    @Override
    public void updateBeforeRender() {
        Component[] messages = hasShiftOrControlDown() ? components64 : components1000;
        for (int i = 0; i < amountButtons.size(); i++) {
            amountButtons.get(i).setMessage(messages[i]);
        }
    }

    private static boolean hasShiftOrControlDown() {
        return Screen.hasShiftDown() || Screen.hasControlDown();
    }

    /**
     * Returns whether the text field begins with an equals sign. This is used by crafting request screens in order to
     * request just enough of an item to bring the total stored amount to the input amount, rather than requesting the
     * input amount itself.
     */
    public boolean startsWithEquals() {
        return textField.getValue().startsWith("=");
    }

    /**
     * Returns the integer value currently in the text-field, if it is a valid number and is within the allowed min/max
     * value.
     */
    public OptionalInt getIntValue() {
        var value = getLongValue();
        if (value.isPresent()) {
            var longValue = value.getAsLong();
            if (longValue > Integer.MAX_VALUE) {
                return OptionalInt.empty();
            }
            return OptionalInt.of((int) longValue);
        }
        return OptionalInt.empty();
    }

    /**
     * Returns the long value currently in the text-field, if it is a valid number and is within the allowed min/max
     * value.
     */
    public OptionalLong getLongValue() {
        double internalValue = getValueInternal();
        if (Double.isNaN(internalValue)) {
            return OptionalLong.empty();
        }

        // Reject decimal values if the unit is integral
        if (type.amountPerUnit() == 1 && hasDecimalPart(internalValue)) {
            return OptionalLong.empty();
        }

        var externalValue = convertToExternalValue(internalValue);
        if (externalValue < minValue) {
            return OptionalLong.empty();
        } else if (externalValue > maxValue) {
            return OptionalLong.empty();
        }
        return OptionalLong.of(externalValue);
    }

    private boolean hasDecimalPart(double value) {
        return value != Math.floor(value);
    }

    public void setLongValue(long value) {
        var internalValue = convertToInternalValue(Longs.constrainToRange(value, minValue, maxValue));
        this.textField.setValue(decimalFormat.format(internalValue));
        this.textField.moveCursorToEnd();
        this.textField.setHighlightPos(0);
        validate();
    }

    private void addQty(long delta) {
        double currentValue = getValueInternal();
        if (Double.isNaN(currentValue)) {
            currentValue = 0;
        }

        double newValue = currentValue + delta;
        double minimum = convertToInternalValue(this.minValue);
        double maximum = convertToInternalValue(this.maxValue);

        if (newValue < minimum) {
            newValue = minimum;
        } else if (newValue > maximum) {
            newValue = maximum;
        } else if (currentValue == 1 && delta > 0 && delta % 10 == 0) {
            newValue = newValue - 1;
        }
        setValueInternal(newValue);
    }

    /**
     * Retrieves the numeric representation of the value entered by the user, if it is convertible.
     */
    private double getValueInternal() {
        var textValue = textField.getValue();
        if (textValue.startsWith("=")) {
            textValue = textValue.substring(1);
        }
        return Math.min(Math.pow(2, 60), NumberUtil.parse(textValue));
    }

    /*
     * Return true if the value entered by the user is a single numeric number and not a mathematical expression
     */
    private boolean isNumber() {
        var position = new ParsePosition(0);
        var textValue = textField.getValue().trim();
        decimalFormat.parse(textValue, position);
        return position.getErrorIndex() == -1 && position.getIndex() == textValue.length();
    }

    /**
     * Changes the value displayed to the user.
     */
    private void setValueInternal(double value) {
        textField.setValue(decimalFormat.format(value));
    }

    private void validate() {
        List<Component> validationErrors = new ArrayList<>();
        List<Component> infoMessages = new ArrayList<>();

        double value = getValueInternal();
        if (!Double.isNaN(value)) {
            // Reject decimal values if the unit is integral
            if (type.amountPerUnit() == 1 && hasDecimalPart(value)) {
                validationErrors.add(GuiText.NumberNonInteger.text());
            } else {
                var externalValue = convertToExternalValue(value);
                if (externalValue < minValue) {
                    var formatted = decimalFormat.format(convertToInternalValue(minValue));
                    validationErrors.add(GuiText.NumberLessThanMinValue.text(formatted));
                } else if (externalValue > maxValue) {
                    var formatted = decimalFormat.format(convertToInternalValue(maxValue));
                    validationErrors.add(GuiText.NumberGreaterThanMaxValue.text(formatted));
                } else if (!isNumber()) { // is a mathematical expression
                    // displaying the evaluation of the expression
                    infoMessages.add(Component.literal("= " + decimalFormat.format(value)));
                }
            }
        } else {
            validationErrors.add(GuiText.InvalidNumber.text());
        }

        boolean valid = validationErrors.isEmpty();
        var tooltip = valid ? infoMessages : validationErrors;
        this.textField.setTextColor(valid ? normalTextColor : errorTextColor);
        this.textField.setTooltipMessage(tooltip);

        if (this.validationIcon != null) {
            this.validationIcon.setValid(valid);
            this.validationIcon.setTooltip(tooltip);
        }
    }

    private Component makeLabel(Component prefix, int amountIndex, boolean useDecimalSteps) {
        return prefix.plainCopy()
                .append(decimalFormat.format(useDecimalSteps ? STEPS_1000[amountIndex] : STEPS_64[amountIndex]));
    }

    public void setHideValidationIcon(boolean hideValidationIcon) {
        this.hideValidationIcon = hideValidationIcon;
    }

    public NumberEntryType getType() {
        return type;
    }

    public void setType(NumberEntryType type) {
        if (this.type == type) {
            return;
        }
        this.type = type;
        setTextFieldBounds(this.textFieldBounds);
        // Update the external with the now changed scaling
        if (onChange != null) {
            onChange.run();
        }

        validate();
    }

    @Override
    public void drawBackgroundLayer(GuiGraphics guiGraphics, Rect2i bounds, Point mouse) {
        if (type.unit() != null) {
            var font = Minecraft.getInstance().font;
            var x = bounds.getX() + textFieldBounds.getX() + textFieldBounds.getWidth()
                    - font.width(type.unit());
            var y = (int) (bounds.getY() + textFieldBounds.getY() + (textFieldBounds.getHeight() - font.lineHeight) / 2f
                    + 1);
            guiGraphics.drawString(font, type.unit(), x, y, ChatFormatting.DARK_GRAY.getColor(), false);
        }
    }

    @Override
    public boolean onMouseWheel(Point mousePos, double delta) {
        if (textFieldBounds.contains(mousePos.getX(), mousePos.getY())) {
            double value = getValueInternal();
            if (!Double.isNaN(value)) {
                if (delta < 0) {
                    addQty(-1);
                } else if (delta > 0) {
                    addQty(1);
                }
                return true;
            }
        }
        return false;
    }

    private long convertToExternalValue(double internalValue) {
        double multiplicand = type.amountPerUnit();
        double value = internalValue * multiplicand;
        return (long) Math.ceil(value);
    }

    private double convertToInternalValue(long externalValue) {
        double divisor = type.amountPerUnit();
        return externalValue / divisor;
    }
}
