/*
 * This file is part of Applied Energistics 2.
 * Copyright (c) 2013 - 2014, AlgorithmX2, All rights reserved.
 *
 * Applied Energistics 2 is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Applied Energistics 2 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Applied Energistics 2.  If not, see <http://www.gnu.org/licenses/lgpl>.
 */

package appeng.client.gui.me.crafting;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;

import org.apache.commons.lang3.time.DurationFormatUtils;

import net.minecraft.ChatFormatting;
import net.minecraft.client.gui.GuiGraphics;
import net.minecraft.client.gui.components.Button;
import net.minecraft.client.gui.screens.Screen;
import net.minecraft.network.chat.Component;
import net.minecraft.world.entity.player.Inventory;

import appeng.api.config.ActionItems;
import appeng.api.config.CpuSelectionMode;
import appeng.api.config.Settings;
import appeng.client.gui.AEBaseScreen;
import appeng.client.gui.StackWithBounds;
import appeng.client.gui.style.ScreenStyle;
import appeng.client.gui.widgets.ActionButton;
import appeng.client.gui.widgets.Scrollbar;
import appeng.client.gui.widgets.ServerSettingToggleButton;
import appeng.client.gui.widgets.SettingToggleButton;
import appeng.core.localization.GuiText;
import appeng.core.sync.network.NetworkHandler;
import appeng.core.sync.packets.BlockHighlightPacket;
import appeng.helpers.CraftExporter;
import appeng.menu.me.crafting.CraftingCPUMenu;
import appeng.menu.me.crafting.CraftingStatus;
import appeng.menu.me.crafting.CraftingStatusEntry;

/**
 * This screen shows the current crafting job that a crafting CPU is working on (if any).
 */
public class CraftingCPUScreen<T extends CraftingCPUMenu> extends AEBaseScreen<T> {

    private final CraftingStatusTableRenderer table;

    private final Button cancel, suspend;

    private final Scrollbar scrollbar;

    private final ActionButton exportCraft;

    private final SettingToggleButton<CpuSelectionMode> schedulingModeButton;

    private CraftingStatus status;

    public CraftingCPUScreen(T menu, Inventory playerInventory, Component title, ScreenStyle style) {
        super(menu, playerInventory, title, style);

        this.table = new CraftingStatusTableRenderer(this, 9, 19);

        this.scrollbar = widgets.addScrollBar("scrollbar");

        this.cancel = this.widgets.addButton("cancel", GuiText.Cancel.text(), menu::cancelCrafting);
        this.suspend = this.widgets.addButton("suspend", GuiText.Suspend.text(), menu::toggleScheduling);

        this.schedulingModeButton = new ServerSettingToggleButton<>(Settings.CPU_SELECTION_MODE,
                CpuSelectionMode.ANY);

        this.exportCraft = this.addToLeftToolbar(new ActionButton(ActionItems.EXPORT_CRAFT, this::exportCraft));
        // This screen is reused for the crafting status in Terminals, where it should not show the config buttons
        if (menu.allowConfiguration()) {
            this.addToLeftToolbar(this.schedulingModeButton);
        }
    }

    @Override
    protected void updateBeforeRender() {
        super.updateBeforeRender();

        // Update the dialog title with an ETA if possible
        Component title = this.getGuiDisplayName(GuiText.CraftingStatus.text());
        if (status != null) {
            final long elapsedTime = status.getElapsedTime();
            final double remainingItems = status.getRemainingItemCount();
            final double startItems = status.getStartItemCount();
            final long eta = (long) (elapsedTime / Math.max(1d, startItems - remainingItems)
                    * remainingItems);

            if (eta > 0 && !getVisualEntries().isEmpty()) {
                final long etaInMilliseconds = TimeUnit.MILLISECONDS.convert(eta, TimeUnit.NANOSECONDS);
                final String etaTimeText = DurationFormatUtils.formatDuration(etaInMilliseconds,
                        GuiText.ETAFormat.getLocal());
                title = title.copy().append(" - " + etaTimeText);
            }
        }
        // Mention that items can't be stored if applicable
        if (menu.isCantStoreItems()) {
            title = title.copy().append(" - ").append(GuiText.CantStoreItems.text().withStyle(ChatFormatting.RED));
        }
        setTextContent(TEXT_ID_DIALOG_TITLE, title);

        final int size = this.status != null ? this.status.getEntries().size() : 0;
        scrollbar.setRange(0, this.table.getScrollableRows(size), 1);

        this.schedulingModeButton.set(this.menu.getSchedulingMode());

        this.exportCraft.visible = !getVisualEntries().isEmpty();
    }

    private List<CraftingStatusEntry> getVisualEntries() {
        return this.status != null ? status.getEntries() : Collections.emptyList();
    }

    @Override
    public void render(GuiGraphics guiGraphics, int mouseX, int mouseY, float btn) {
        this.cancel.active = !getVisualEntries().isEmpty();
        this.suspend.active = this.cancel.active;

        super.render(guiGraphics, mouseX, mouseY, btn);
    }

    @Override
    public void drawFG(GuiGraphics guiGraphics, int offsetX, int offsetY, int mouseX, int mouseY) {
        super.drawFG(guiGraphics, offsetX, offsetY, mouseX, mouseY);

        if (status != null) {
            this.table.render(guiGraphics, mouseX, mouseY, status.getEntries(), scrollbar.getCurrentScroll());
        }
    }

    @org.jetbrains.annotations.Nullable
    @Override
    public StackWithBounds getStackUnderMouse(double mouseX, double mouseY) {
        var hovered = table.getHoveredStack();
        if (hovered != null) {
            return hovered;
        }
        return super.getStackUnderMouse(mouseX, mouseY);
    }

    public void postUpdate(CraftingStatus status) {
        Map<Long, CraftingStatusEntry> entries;
        if (this.status == null || status.isFullStatus()) {
            // Start from scratch.
            // We can't just reuse the status because we have to filter out deleted entries.
            entries = new LinkedHashMap<>();
        } else {
            // Merge the status entries.
            entries = new LinkedHashMap<>(this.status.getEntries().size());
            for (CraftingStatusEntry entry : this.status.getEntries()) {
                entries.put(entry.getSerial(), entry);
            }
        }

        for (CraftingStatusEntry entry : status.getEntries()) {
            if (entry.isDeleted()) {
                entries.remove(entry.getSerial());
                continue;
            }

            entries.merge(entry.getSerial(), entry, (existingEntry, newEntry) -> new CraftingStatusEntry(
                    existingEntry.getSerial(),
                    existingEntry.getWhat(),
                    newEntry.getStoredAmount(),
                    newEntry.getActiveAmount(),
                    newEntry.getPendingAmount()));
        }

        List<CraftingStatusEntry> sortedEntries = new ArrayList<>(entries.values());
        Collections.sort(sortedEntries);
        this.status = new CraftingStatus(
                true,
                status.getElapsedTime(),
                status.getRemainingItemCount(),
                status.getStartItemCount(),
                sortedEntries,
                status.isSuspended());
        this.suspend.setMessage(status.isSuspended() ? GuiText.Resume.text() : GuiText.Suspend.text());
    }

    @Override
    public boolean mouseClicked(double xCoord, double yCoord, int btn) {
        if (btn == 0 && Screen.hasShiftDown()) {
            var hovered = table.getHoveredStack();
            if (hovered != null) {
                var packet = new BlockHighlightPacket.HighlightWhat(hovered.stack().what());
                NetworkHandler.instance().sendToServer(packet);
                return true;
            }
        }
        return super.mouseClicked(xCoord, yCoord, btn);
    }

    protected void exportCraft() {
        var exportObject = new JsonObject();
        var currentStatus = new JsonObject();
        currentStatus.addProperty("elapsedTime", status.getElapsedTime());
        currentStatus.addProperty("remainingItemCount", status.getRemainingItemCount());
        currentStatus.addProperty("startItemCount", status.getStartItemCount());
        currentStatus.addProperty("suspended", status.isSuspended());
        exportObject.add("status", currentStatus);
        var entryArray = new JsonArray();
        for (var entry : status.getEntries()) {
            var entryObject = new JsonObject();
            entryObject.addProperty("what", entry.getWhat().getId().toString());
            entryObject.addProperty("serial", entry.getSerial());
            entryObject.addProperty("storedAmount", entry.getStoredAmount());
            entryObject.addProperty("activeAmount", entry.getActiveAmount());
            entryObject.addProperty("pendingAmount", entry.getPendingAmount());
            entryArray.add(entryObject);
        }
        exportObject.add("entries", entryArray);
        CraftExporter.exportCraft(exportObject, getPlayer(), CraftExporter.ExportType.CRAFTING_STATUS);
    }
}
