/*
 * This file is part of Applied Energistics 2.
 * Copyright (c) 2013 - 2015, AlgorithmX2, All rights reserved.
 *
 * Applied Energistics 2 is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Applied Energistics 2 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Applied Energistics 2.  If not, see <http://www.gnu.org/licenses/lgpl>.
 */

package appeng.items.contents;

import java.util.Objects;
import java.util.function.BiConsumer;

import com.google.common.base.Preconditions;

import org.jetbrains.annotations.Nullable;

import net.minecraft.world.entity.player.Player;
import net.minecraft.world.inventory.AbstractContainerMenu;
import net.minecraft.world.item.ItemStack;

import appeng.api.config.Actionable;
import appeng.api.config.PowerMultiplier;
import appeng.api.config.Settings;
import appeng.api.config.SortDir;
import appeng.api.config.SortOrder;
import appeng.api.config.ViewItems;
import appeng.api.features.HotkeyAction;
import appeng.api.implementations.menuobjects.IPortableTerminal;
import appeng.api.implementations.menuobjects.ItemMenuHost;
import appeng.api.stacks.AEKeyType;
import appeng.api.storage.MEStorage;
import appeng.api.storage.StorageCells;
import appeng.api.storage.cells.IBasicCellItem;
import appeng.api.util.IConfigManager;
import appeng.items.tools.powered.AbstractPortableCell;
import appeng.menu.ISubMenu;
import appeng.util.ConfigManager;

/**
 * Hosts the terminal interface for a {@link appeng.items.tools.powered.AbstractPortableCell}.
 */
public class PortableCellMenuHost extends ItemMenuHost implements IPortableTerminal {
    private final BiConsumer<Player, ISubMenu> returnMainMenu;
    private final MEStorage cellStorage;
    private final AbstractPortableCell item;

    public PortableCellMenuHost(Player player, @Nullable Integer slot, AbstractPortableCell item, ItemStack itemStack,
            BiConsumer<Player, ISubMenu> returnMainMenu) {
        super(player, slot, itemStack);
        Preconditions.checkArgument(itemStack.getItem() == item, "Stack doesn't match item");
        this.returnMainMenu = returnMainMenu;
        this.cellStorage = StorageCells.getCellInventory(itemStack, null);
        Objects.requireNonNull(cellStorage, "Portable cell doesn't expose a cell inventory.");
        this.item = item;
    }

    @Override
    public boolean onBroadcastChanges(AbstractContainerMenu menu) {
        return ensureItemStillInSlot() && drainPower();
    }

    @Override
    public double extractAEPower(double amt, Actionable mode, PowerMultiplier usePowerMultiplier) {
        amt = usePowerMultiplier.multiply(amt);

        if (mode == Actionable.SIMULATE) {
            return usePowerMultiplier.divide(Math.min(amt, this.item.getAECurrentPower(getItemStack())));
        }

        return usePowerMultiplier.divide(this.item.extractAEPower(getItemStack(), amt, Actionable.MODULATE));
    }

    @Override
    public MEStorage getInventory() {
        return cellStorage;
    }

    @Override
    public IConfigManager getConfigManager() {
        var out = new ConfigManager((manager, settingName) -> {
            manager.writeToNBT(getItemStack().getOrCreateTag());
        });

        out.registerSetting(Settings.SORT_BY, SortOrder.NAME);
        out.registerSetting(Settings.VIEW_MODE, ViewItems.ALL);
        out.registerSetting(Settings.SORT_DIRECTION, SortDir.ASCENDING);

        out.readFromNBT(getItemStack().getOrCreateTag().copy());
        return out;
    }

    @Override
    public void returnToMainMenu(Player player, ISubMenu subMenu) {
        returnMainMenu.accept(player, subMenu);
    }

    @Override
    public ItemStack getMainMenuIcon() {
        return getItemStack();
    }

    public String getCloseHotkey() {
        if (item instanceof IBasicCellItem cellItem) {
            if (cellItem.getKeyType().equals(AEKeyType.items())) {
                return HotkeyAction.PORTABLE_ITEM_CELL;
            } else if (cellItem.getKeyType().equals(AEKeyType.fluids())) {
                return HotkeyAction.PORTABLE_FLUID_CELL;
            }
        }

        return null; // We don't know
    }
}
